# ============================================================
# NZG 73 - System Controller Backend
# اردو کمنٹ: یہ مین پائتھون فائل ہے جو سرور چلاتی ہے
# ============================================================

# اردو کمنٹ: ضروری لائبریریز امپورٹ کر رہے ہیں
import os
import sys
import json
import socket
import ctypes
import subprocess
import threading
import winreg  # اردو کمنٹ: ونڈوز رجسٹری کے لیے
from pathlib import Path

# اردو کمنٹ: Flask ویب فریم ورک
from flask import Flask, jsonify, send_from_directory, request
from flask_cors import CORS

# اردو کمنٹ: بیٹری اور پروسیس کی معلومات کے لیے
import psutil

# ============================================================
# اردو کمنٹ: بنیادی سیٹنگز
# ============================================================

# اردو کمنٹ: پورٹ نمبر جس پر سرور چلے گا
PORT = 21880

# اردو کمنٹ: ایپلیکیشن کا نام
APP_NAME = "NZG73"

# اردو کمنٹ: سیٹنگز فائل کا راستہ (اسٹیٹ یاد رکھنے کے لیے)
SETTINGS_FILE = os.path.join(os.path.dirname(os.path.abspath(__file__)), "nzg73_settings.json")

# اردو کمنٹ: HTML فولڈر کا راستہ
HTML_FOLDER = os.path.join(os.path.dirname(os.path.abspath(__file__)), "Html")

# ============================================================
# اردو کمنٹ: Flask ایپ بنانا
# ============================================================

app = Flask(__name__, static_folder=HTML_FOLDER)
CORS(app)  # اردو کمنٹ: کراس اوریجن ریکویسٹس کی اجازت

# ============================================================
# اردو کمنٹ: سیٹنگز لوڈ اور سیو کرنے کے فنکشنز
# ============================================================

def load_settings():
    """
    اردو کمنٹ: سیٹنگز فائل سے ڈیٹا لوڈ کرتا ہے
    اگر فائل نہیں ہے تو ڈیفالٹ سیٹنگز دیتا ہے
    """
    default_settings = {
        "screen_awake": False,         # اردو کمنٹ: اسکرین آن رکھنا
        "charging_enabled": True,      # اردو کمنٹ: چارجنگ آن/آف
        "autostart_enabled": False,    # اردو کمنٹ: آٹو اسٹارٹ
        "monitor_on": True,            # اردو کمنٹ: مانیٹر آن/آف
        "console_state": "show",       # اردو کمنٹ: کنسول سٹیٹ (show/hide/kill)
        "computer_kill_done": False    # اردو کمنٹ: کمپیوٹر کل ہوا یا نہیں
    }
    
    try:
        if os.path.exists(SETTINGS_FILE):
            with open(SETTINGS_FILE, 'r') as f:
                return json.load(f)
    except Exception as e:
        print(f"Settings load error: {e}")
    
    return default_settings


def save_settings(settings):
    """
    اردو کمنٹ: سیٹنگز کو فائل میں سیو کرتا ہے
    تاکہ ری اسٹارٹ کے بعد بھی یاد رہیں
    """
    try:
        with open(SETTINGS_FILE, 'w') as f:
            json.dump(settings, f, indent=4)
        return True
    except Exception as e:
        print(f"Settings save error: {e}")
        return False


# اردو کمنٹ: گلوبل سیٹنگز لوڈ کرنا
settings = load_settings()

# ============================================================
# اردو کمنٹ: اسکرین اویک کے لیے متغیرات اور فنکشنز
# ============================================================

# اردو کمنٹ: ونڈوز API کانسٹنٹس
ES_CONTINUOUS = 0x80000000
ES_SYSTEM_REQUIRED = 0x00000001
ES_DISPLAY_REQUIRED = 0x00000002

# اردو کمنٹ: اسکرین اویک تھریڈ کنٹرول
screen_awake_active = False
screen_awake_thread = None


def keep_screen_awake():
    """
    اردو کمنٹ: یہ فنکشن اسکرین کو آن رکھتا ہے
    ونڈوز API استعمال کرتے ہوئے سسٹم کو سلیپ سے روکتا ہے
    """
    global screen_awake_active
    
    while screen_awake_active:
        # اردو کمنٹ: ونڈوز کو بتانا کہ سسٹم اور ڈسپلے ایکٹو رکھو
        ctypes.windll.kernel32.SetThreadExecutionState(
            ES_CONTINUOUS | ES_SYSTEM_REQUIRED | ES_DISPLAY_REQUIRED
        )
        # اردو کمنٹ: ہر 30 سیکنڈ بعد دوبارہ سیٹ کرنا
        threading.Event().wait(30)
    
    # اردو کمنٹ: نارمل سٹیٹ واپس کرنا
    ctypes.windll.kernel32.SetThreadExecutionState(ES_CONTINUOUS)


def start_screen_awake():
    """
    اردو کمنٹ: اسکرین اویک شروع کرنے کا فنکشن
    نیا تھریڈ بناتا ہے تاکہ بیک گراؤنڈ میں چلے
    """
    global screen_awake_active, screen_awake_thread
    
    if not screen_awake_active:
        screen_awake_active = True
        screen_awake_thread = threading.Thread(target=keep_screen_awake, daemon=True)
        screen_awake_thread.start()
        
        # اردو کمنٹ: سیٹنگز سیو کرنا
        settings["screen_awake"] = True
        save_settings(settings)
        
        print("Screen Awake: ENABLED")
        return True
    return False


def stop_screen_awake():
    """
    اردو کمنٹ: اسکرین اویک بند کرنے کا فنکشن
    """
    global screen_awake_active
    
    screen_awake_active = False
    
    # اردو کمنٹ: سیٹنگز سیو کرنا
    settings["screen_awake"] = False
    save_settings(settings)
    
    # اردو کمنٹ: نارمل سٹیٹ واپس کرنا
    ctypes.windll.kernel32.SetThreadExecutionState(ES_CONTINUOUS)
    
    print("Screen Awake: DISABLED")
    return True


# ============================================================
# اردو کمنٹ: آٹو اسٹارٹ کے فنکشنز (ونڈوز رجسٹری)
# ============================================================

def get_script_path():
    """
    اردو کمنٹ: موجودہ اسکرپٹ کا پورا راستہ واپس کرتا ہے
    """
    return os.path.abspath(__file__)


def enable_autostart():
    """
    اردو کمنٹ: ونڈوز سٹارٹ اپ میں پروگرام شامل کرتا ہے
    رجسٹری میں انٹری بناتا ہے
    """
    try:
        # اردو کمنٹ: رجسٹری کی کلید کھولنا
        key = winreg.OpenKey(
            winreg.HKEY_CURRENT_USER,
            r"Software\Microsoft\Windows\CurrentVersion\Run",
            0,
            winreg.KEY_SET_VALUE
        )
        
        # اردو کمنٹ: پائتھون اور اسکرپٹ کا راستہ
        python_exe = sys.executable
        script_path = get_script_path()
        
        # اردو کمنٹ: کمانڈ بنانا (پوشیدہ موڈ میں چلانے کے لیے)
        command = f'"{python_exe}" "{script_path}"'
        
        # اردو کمنٹ: رجسٹری میں ویلیو سیٹ کرنا
        winreg.SetValueEx(key, APP_NAME, 0, winreg.REG_SZ, command)
        winreg.CloseKey(key)
        
        # اردو کمنٹ: سیٹنگز سیو کرنا
        settings["autostart_enabled"] = True
        save_settings(settings)
        
        print("Auto-Start: ENABLED")
        return True
        
    except Exception as e:
        print(f"Autostart enable error: {e}")
        return False


def disable_autostart():
    """
    اردو کمنٹ: ونڈوز سٹارٹ اپ سے پروگرام ہٹاتا ہے
    """
    try:
        # اردو کمنٹ: رجسٹری کی کلید کھولنا
        key = winreg.OpenKey(
            winreg.HKEY_CURRENT_USER,
            r"Software\Microsoft\Windows\CurrentVersion\Run",
            0,
            winreg.KEY_SET_VALUE
        )
        
        # اردو کمنٹ: انٹری ڈیلیٹ کرنا
        try:
            winreg.DeleteValue(key, APP_NAME)
        except FileNotFoundError:
            pass  # اردو کمنٹ: اگر پہلے سے نہیں ہے تو کوئی بات نہیں
            
        winreg.CloseKey(key)
        
        # اردو کمنٹ: سیٹنگز سیو کرنا
        settings["autostart_enabled"] = False
        save_settings(settings)
        
        print("Auto-Start: DISABLED")
        return True
        
    except Exception as e:
        print(f"Autostart disable error: {e}")
        return False


def check_autostart():
    """
    اردو کمنٹ: چیک کرتا ہے کہ آٹو اسٹارٹ آن ہے یا نہیں
    """
    try:
        key = winreg.OpenKey(
            winreg.HKEY_CURRENT_USER,
            r"Software\Microsoft\Windows\CurrentVersion\Run",
            0,
            winreg.KEY_READ
        )
        
        try:
            winreg.QueryValueEx(key, APP_NAME)
            winreg.CloseKey(key)
            return True
        except FileNotFoundError:
            winreg.CloseKey(key)
            return False
            
    except Exception:
        return False


# ============================================================
# اردو کمنٹ: مانیٹر کنٹرول کے فنکشنز
# ============================================================

def turn_monitor_off():
    """
    اردو کمنٹ: مانیٹر/اسکرین بند کرتا ہے
    """
    try:
        # اردو کمنٹ: ونڈوز API سے مانیٹر آف کرنا
        ctypes.windll.user32.SendMessageW(0xFFFF, 0x0112, 0xF170, 2)
        
        settings["monitor_on"] = False
        save_settings(settings)
        
        print("Monitor: OFF")
        return True
    except Exception as e:
        print(f"Monitor off error: {e}")
        return False


def turn_monitor_on():
    """
    اردو کمنٹ: مانیٹر/اسکرین آن کرتا ہے
    """
    try:
        # اردو کمنٹ: ماؤس ہلا کر مانیٹر جگانا
        ctypes.windll.user32.mouse_event(0x0001, 0, 0, 0, 0)
        
        settings["monitor_on"] = True
        save_settings(settings)
        
        print("Monitor: ON")
        return True
    except Exception as e:
        print(f"Monitor on error: {e}")
        return False


# ============================================================
# اردو کمنٹ: کنسول کنٹرول کے فنکشنز (Hide/Show/Kill)
# ============================================================

def get_current_process_id():
    """
    اردو کمنٹ: موجودہ پروسیس کی ID واپس کرتا ہے
    تاکہ اسے بند نہ کیا جائے
    """
    return os.getpid()


def hide_all_windows():
    """
    اردو کمنٹ: تمام ونڈوز کو minimize کرتا ہے
    """
    try:
        # اردو کمنٹ: Win+D دبانے جیسا اثر
        ctypes.windll.user32.keybd_event(0x5B, 0, 0, 0)  # Win key down
        ctypes.windll.user32.keybd_event(0x44, 0, 0, 0)  # D key down
        ctypes.windll.user32.keybd_event(0x44, 0, 2, 0)  # D key up
        ctypes.windll.user32.keybd_event(0x5B, 0, 2, 0)  # Win key up
        
        settings["console_state"] = "hide"
        save_settings(settings)
        
        print("Console: HIDDEN")
        return True
    except Exception as e:
        print(f"Hide windows error: {e}")
        return False


def show_all_windows():
    """
    اردو کمنٹ: تمام ونڈوز کو واپس لاتا ہے
    """
    try:
        # اردو کمنٹ: Win+D دوبارہ دبانے سے ونڈوز واپس آتی ہیں
        ctypes.windll.user32.keybd_event(0x5B, 0, 0, 0)  # Win key down
        ctypes.windll.user32.keybd_event(0x44, 0, 0, 0)  # D key down
        ctypes.windll.user32.keybd_event(0x44, 0, 2, 0)  # D key up
        ctypes.windll.user32.keybd_event(0x5B, 0, 2, 0)  # Win key up
        
        settings["console_state"] = "show"
        save_settings(settings)
        
        print("Console: SHOWN")
        return True
    except Exception as e:
        print(f"Show windows error: {e}")
        return False


def kill_console_processes():
    """
    اردو کمنٹ: CMD اور Python پروسیسز بند کرتا ہے
    لیکن موجودہ پروسیس (جو Web UI چلا رہا ہے) کو نہیں
    """
    current_pid = get_current_process_id()
    killed_count = 0
    
    try:
        for proc in psutil.process_iter(['pid', 'name']):
            try:
                pinfo = proc.info
                name = pinfo['name'].lower()
                pid = pinfo['pid']
                
                # اردو کمنٹ: موجودہ پروسیس کو چھوڑنا
                if pid == current_pid:
                    continue
                
                # اردو کمنٹ: CMD اور Python پروسیسز بند کرنا
                if any(x in name for x in ['cmd.exe', 'powershell.exe', 'python.exe', 'conhost.exe']):
                    try:
                        proc.terminate()
                        killed_count += 1
                    except:
                        pass
                        
            except (psutil.NoSuchProcess, psutil.AccessDenied):
                continue
        
        settings["console_state"] = "kill"
        save_settings(settings)
        
        print(f"Console Kill: {killed_count} processes killed")
        return killed_count
        
    except Exception as e:
        print(f"Console kill error: {e}")
        return 0


# ============================================================
# اردو کمنٹ: کمپیوٹر کل (غیر ضروری پروگرامز بند کرنا)
# ============================================================

def kill_unnecessary_programs():
    """
    اردو کمنٹ: Chrome, VLC, Video Recorder وغیرہ بند کرتا ہے
    Python اور CMD کو نہیں چھیڑتا
    """
    current_pid = get_current_process_id()
    killed_count = 0
    
    # اردو کمنٹ: جو پروگرامز بند کرنے ہیں
    programs_to_kill = [
        'chrome.exe', 'firefox.exe', 'msedge.exe', 'opera.exe', 'brave.exe',
        'vlc.exe', 'wmplayer.exe', 'spotify.exe',
        'obs64.exe', 'obs32.exe', 'obs.exe',
        'zoom.exe', 'teams.exe', 'skype.exe',
        'notepad.exe', 'wordpad.exe',
        'winword.exe', 'excel.exe', 'powerpnt.exe',
        'acrobat.exe', 'acrord32.exe',
        'photoshop.exe', 'illustrator.exe',
        'discord.exe', 'telegram.exe', 'whatsapp.exe'
    ]
    
    # اردو کمنٹ: جو پروگرامز نہیں بند کرنے (محفوظ رکھنے ہیں)
    safe_programs = ['python.exe', 'pythonw.exe', 'cmd.exe', 'powershell.exe', 'conhost.exe', 'explorer.exe']
    
    try:
        for proc in psutil.process_iter(['pid', 'name']):
            try:
                pinfo = proc.info
                name = pinfo['name'].lower()
                pid = pinfo['pid']
                
                # اردو کمنٹ: موجودہ پروسیس اور محفوظ پروگرامز کو چھوڑنا
                if pid == current_pid:
                    continue
                    
                if name in safe_programs:
                    continue
                
                # اردو کمنٹ: لسٹ میں موجود پروگرامز بند کرنا
                if name in programs_to_kill:
                    try:
                        proc.terminate()
                        killed_count += 1
                    except:
                        pass
                        
            except (psutil.NoSuchProcess, psutil.AccessDenied):
                continue
        
        settings["computer_kill_done"] = True
        save_settings(settings)
        
        print(f"Computer Kill: {killed_count} programs closed")
        return killed_count
        
    except Exception as e:
        print(f"Computer kill error: {e}")
        return 0


# ============================================================
# اردو کمنٹ: نیٹ ورک کے فنکشنز
# ============================================================

def get_local_ip():
    """
    اردو کمنٹ: لوکل نیٹ ورک آئی پی ایڈریس حاصل کرتا ہے
    """
    try:
        # اردو کمنٹ: ڈمی کنکشن بنا کر آئی پی حاصل کرنا
        s = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
        s.connect(("8.8.8.8", 80))
        ip = s.getsockname()[0]
        s.close()
        return ip
    except Exception:
        return "127.0.0.1"


# ============================================================
# اردو کمنٹ: API روٹس - مین پیج
# ============================================================

@app.route('/')
def index():
    """
    اردو کمنٹ: مین پیج سرو کرتا ہے
    Html فولڈر سے index.html دیتا ہے
    """
    return send_from_directory(HTML_FOLDER, 'index.html')


@app.route('/<path:filename>')
def serve_static(filename):
    """
    اردو کمنٹ: سٹیٹک فائلز سرو کرتا ہے
    """
    return send_from_directory(HTML_FOLDER, filename)


# ============================================================
# اردو کمنٹ: API روٹس - بیٹری
# ============================================================

@app.route('/api/battery')
def get_battery():
    """
    اردو کمنٹ: بیٹری کی معلومات واپس کرتا ہے
    پرسنٹیج اور چارجنگ سٹیٹس
    """
    try:
        battery = psutil.sensors_battery()
        
        if battery:
            return jsonify({
                "percent": battery.percent,
                "charging": battery.power_plugged,
                "charging_enabled": settings.get("charging_enabled", True),
                "time_left": battery.secsleft if battery.secsleft > 0 else None
            })
        else:
            # اردو کمنٹ: ڈیسک ٹاپ پی سی کے لیے (بیٹری نہیں ہوتی)
            return jsonify({
                "percent": 100,
                "charging": True,
                "charging_enabled": True,
                "time_left": None,
                "message": "Desktop PC - No battery"
            })
            
    except Exception as e:
        return jsonify({"error": str(e)}), 500


@app.route('/api/charging/enable', methods=['POST'])
def enable_charging():
    """
    اردو کمنٹ: چارجنگ انیبل کرتا ہے
    نوٹ: زیادہ تر لیپ ٹاپس میں سافٹ ویئر سے چارجنگ کنٹرول ممکن نہیں
    """
    settings["charging_enabled"] = True
    save_settings(settings)
    
    # اردو کمنٹ: کچھ لیپ ٹاپس (جیسے Lenovo, ASUS) کے لیے خاص کمانڈز ہیں
    # یہاں ہم صرف سٹیٹ سیو کر رہے ہیں
    
    return jsonify({"success": True, "message": "Charging enabled (if supported)"})


@app.route('/api/charging/disable', methods=['POST'])
def disable_charging():
    """
    اردو کمنٹ: چارجنگ ڈس ایبل کرتا ہے
    """
    settings["charging_enabled"] = False
    save_settings(settings)
    
    return jsonify({"success": True, "message": "Charging disabled (if supported)"})


# ============================================================
# اردو کمنٹ: API روٹس - اسکرین اویک
# ============================================================

@app.route('/api/screen/status')
def screen_status():
    """
    اردو کمنٹ: اسکرین اویک کی موجودہ حالت واپس کرتا ہے
    """
    return jsonify({
        "awake": settings.get("screen_awake", False)
    })


@app.route('/api/screen/on', methods=['POST'])
def screen_on():
    """
    اردو کمنٹ: اسکرین اویک آن کرتا ہے
    """
    start_screen_awake()
    return jsonify({"success": True, "awake": True})


@app.route('/api/screen/off', methods=['POST'])
def screen_off():
    """
    اردو کمنٹ: اسکرین اویک آف کرتا ہے
    """
    stop_screen_awake()
    return jsonify({"success": True, "awake": False})


# ============================================================
# اردو کمنٹ: API روٹس - آٹو اسٹارٹ
# ============================================================

@app.route('/api/autostart/status')
def autostart_status():
    """
    اردو کمنٹ: آٹو اسٹارٹ کی حالت چیک کرتا ہے
    """
    return jsonify({
        "enabled": check_autostart()
    })


@app.route('/api/autostart/enable', methods=['POST'])
def autostart_enable():
    """
    اردو کمنٹ: آٹو اسٹارٹ آن کرتا ہے
    """
    success = enable_autostart()
    return jsonify({"success": success})


@app.route('/api/autostart/disable', methods=['POST'])
def autostart_disable():
    """
    اردو کمنٹ: آٹو اسٹارٹ آف کرتا ہے
    """
    success = disable_autostart()
    return jsonify({"success": success})


# ============================================================
# اردو کمنٹ: API روٹس - نیٹ ورک
# ============================================================

@app.route('/api/network')
def network_info():
    """
    اردو کمنٹ: نیٹ ورک کی معلومات واپس کرتا ہے
    دوسرے ڈیوائسز کے لیے URL
    """
    ip = get_local_ip()
    url = f"http://{ip}:{PORT}"
    
    return jsonify({
        "ip": ip,
        "port": PORT,
        "url": url
    })


# ============================================================
# اردو کمنٹ: API روٹس - CMD پروسیسز
# ============================================================

@app.route('/api/processes')
def get_processes():
    """
    اردو کمنٹ: چلتے ہوئے CMD پروسیسز کی لسٹ واپس کرتا ہے
    """
    processes = []
    
    try:
        for proc in psutil.process_iter(['pid', 'name', 'cmdline']):
            try:
                pinfo = proc.info
                name = pinfo['name'].lower()
                
                # اردو کمنٹ: صرف CMD اور کنسول پروسیسز دکھانا
                if any(x in name for x in ['cmd', 'powershell', 'python', 'node', 'conhost']):
                    processes.append({
                        "pid": pinfo['pid'],
                        "name": pinfo['name'],
                        "cmdline": ' '.join(pinfo['cmdline'] or [])[:100]
                    })
            except (psutil.NoSuchProcess, psutil.AccessDenied):
                continue
                
    except Exception as e:
        print(f"Process list error: {e}")
    
    return jsonify(processes)


@app.route('/api/process/kill/<int:pid>', methods=['POST'])
def kill_process(pid):
    """
    اردو کمنٹ: مخصوص پروسیس کو بند کرتا ہے
    """
    try:
        proc = psutil.Process(pid)
        proc.terminate()
        
        # اردو کمنٹ: 3 سیکنڈ انتظار کرنا
        proc.wait(timeout=3)
        
        return jsonify({"success": True, "message": f"Process {pid} killed"})
        
    except psutil.NoSuchProcess:
        return jsonify({"success": False, "message": "Process not found"}), 404
        
    except psutil.AccessDenied:
        # اردو کمنٹ: زبردستی بند کرنا
        try:
            subprocess.run(['taskkill', '/F', '/PID', str(pid)], capture_output=True)
            return jsonify({"success": True, "message": f"Process {pid} force killed"})
        except:
            return jsonify({"success": False, "message": "Access denied"}), 403
            
    except Exception as e:
        return jsonify({"success": False, "message": str(e)}), 500


# ============================================================
# اردو کمنٹ: API روٹس - سسٹم پاور
# ============================================================

@app.route('/api/power/restart', methods=['POST'])
def restart_pc():
    """
    اردو کمنٹ: پی سی کو ری اسٹارٹ کرتا ہے
    """
    try:
        # اردو کمنٹ: 5 سیکنڈ بعد ری اسٹارٹ
        subprocess.Popen(['shutdown', '/r', '/t', '5'])
        return jsonify({"success": True, "message": "Restarting in 5 seconds..."})
    except Exception as e:
        return jsonify({"success": False, "message": str(e)}), 500


@app.route('/api/power/shutdown', methods=['POST'])
def shutdown_pc():
    """
    اردو کمنٹ: پی سی کو شٹ ڈاؤن کرتا ہے
    """
    try:
        # اردو کمنٹ: 5 سیکنڈ بعد شٹ ڈاؤن
        subprocess.Popen(['shutdown', '/s', '/t', '5'])
        return jsonify({"success": True, "message": "Shutting down in 5 seconds..."})
    except Exception as e:
        return jsonify({"success": False, "message": str(e)}), 500


# ============================================================
# اردو کمنٹ: API روٹس - مانیٹر کنٹرول
# ============================================================

@app.route('/api/monitor/status')
def monitor_status():
    """
    اردو کمنٹ: مانیٹر کی موجودہ حالت واپس کرتا ہے
    """
    return jsonify({
        "on": settings.get("monitor_on", True)
    })


@app.route('/api/monitor/on', methods=['POST'])
def monitor_on():
    """
    اردو کمنٹ: مانیٹر آن کرتا ہے
    """
    success = turn_monitor_on()
    return jsonify({"success": success, "on": True})


@app.route('/api/monitor/off', methods=['POST'])
def monitor_off():
    """
    اردو کمنٹ: مانیٹر آف کرتا ہے
    """
    success = turn_monitor_off()
    return jsonify({"success": success, "on": False})


# ============================================================
# اردو کمنٹ: API روٹس - کنسول کنٹرول
# ============================================================

@app.route('/api/console/status')
def console_status():
    """
    اردو کمنٹ: کنسول کی موجودہ حالت واپس کرتا ہے
    """
    return jsonify({
        "state": settings.get("console_state", "show")
    })


@app.route('/api/console/hide', methods=['POST'])
def console_hide():
    """
    اردو کمنٹ: تمام ونڈوز چھپاتا ہے
    """
    success = hide_all_windows()
    return jsonify({"success": success, "state": "hide"})


@app.route('/api/console/show', methods=['POST'])
def console_show():
    """
    اردو کمنٹ: تمام ونڈوز دکھاتا ہے
    """
    success = show_all_windows()
    return jsonify({"success": success, "state": "show"})


@app.route('/api/console/kill', methods=['POST'])
def console_kill():
    """
    اردو کمنٹ: CMD/Python پروسیسز بند کرتا ہے (موجودہ کو چھوڑ کر)
    """
    count = kill_console_processes()
    return jsonify({"success": True, "killed": count, "state": "kill"})


# ============================================================
# اردو کمنٹ: API روٹس - کمپیوٹر کل
# ============================================================

@app.route('/api/computer/kill/status')
def computer_kill_status():
    """
    اردو کمنٹ: کمپیوٹر کل کی حالت واپس کرتا ہے
    """
    return jsonify({
        "done": settings.get("computer_kill_done", False)
    })


@app.route('/api/computer/kill', methods=['POST'])
def computer_kill():
    """
    اردو کمنٹ: غیر ضروری پروگرامز بند کرتا ہے
    """
    count = kill_unnecessary_programs()
    return jsonify({"success": True, "killed": count})


@app.route('/api/computer/kill/reset', methods=['POST'])
def computer_kill_reset():
    """
    اردو کمنٹ: کمپیوٹر کل سٹیٹس ری سیٹ کرتا ہے
    """
    settings["computer_kill_done"] = False
    save_settings(settings)
    return jsonify({"success": True, "done": False})


# ============================================================
# اردو کمنٹ: سٹارٹ اپ پر سیٹنگز ریسٹور کرنا
# ============================================================

def restore_settings_on_startup():
    """
    اردو کمنٹ: جب پروگرام شروع ہو تو پچھلی سیٹنگز واپس لانا
    """
    global settings
    settings = load_settings()
    
    # اردو کمنٹ: اگر اسکرین اویک آن تھا تو دوبارہ شروع کرنا
    if settings.get("screen_awake", False):
        print("Restoring Screen Awake state...")
        start_screen_awake()


# ============================================================
# اردو کمنٹ: مین فنکشن - پروگرام یہاں سے شروع ہوتا ہے
# ============================================================

def main():
    """
    اردو کمنٹ: یہ مین فنکشن ہے جو سرور شروع کرتا ہے
    """
    # اردو کمنٹ: بینر پرنٹ کرنا
    print("=" * 50)
    print("   NZG 73 - System Controller")
    print("=" * 50)
    print()
    
    # اردو کمنٹ: پچھلی سیٹنگز ریسٹور کرنا
    restore_settings_on_startup()
    
    # اردو کمنٹ: آئی پی اور پورٹ دکھانا
    local_ip = get_local_ip()
    print(f"   Local:   http://127.0.0.1:{PORT}")
    print(f"   Network: http://{local_ip}:{PORT}")
    print()
    print("   Press Ctrl+C to stop")
    print("=" * 50)
    print()
    
    # اردو کمنٹ: سرور شروع کرنا
    # host='0.0.0.0' کا مطلب ہے کہ سب آئی پیز پر سنے گا
    # threaded=True کا مطلب ہے کہ ایک سے زیادہ ریکویسٹس ہینڈل کر سکے
    app.run(
        host='0.0.0.0',
        port=PORT,
        debug=False,
        threaded=True,
        use_reloader=False
    )


# اردو کمنٹ: اگر یہ فائل براہ راست چلائی جائے تو مین فنکشن چلے
if __name__ == '__main__':
    main()
